// Translations
const translations = {
    es: {
        'processing-title': 'Pago con Mercado Pago',
        'processing-subtitle': 'Transferencia',
        'recipient': 'Destinatario',
        'account': 'CVU/CBU',
        'amount': 'Importe exacto',
        'after-payment': 'Después del pago, tocá',
        'after-payment-pagado': '«He pagado»:',
        'paid-button': 'He pagado',
        'time-to-pay': 'Tiempo para pagar',
        'confirm-title': 'Atención',
        'confirm-message': 'Verifica los datos y confirma si pagaste correctamente',
        'amount-label': 'Monto del pago',
        'phone-label': 'Teléfono del destinatario',
        'bank-label': 'Banco',
        'recipient-full': 'Nombre completo del destinatario',
        'back-button': 'Volver al pago',
        'confirm-button': 'Confirmar',
        'searching-title': 'Estamos buscando tu transferencia',
        'searching-message': 'El proceso puede llevar hasta 20 minutos.',
        'back-to-store': 'Volver a la tienda',
        'back-to-payment': 'Volver al pago',
        'payment-details': 'Detalles del pago',
        'detail-id': 'ID',
        'detail-bank': 'Datos bancarios',
        'detail-method': 'Método de pago',
        'detail-amount': 'Monto',
        'detail-recipient': 'Destinatario',
        'detail-bank-name': 'Banco del destinatario',
        'detail-date': 'Fecha de creación',
        'not-credited': 'No acreditado',
        'expired-message': 'Cancelaste el pago, o el tiempo de pago ha vencido',
        'appeal-button': 'Apelar pago',
        'appeal-text': 'Apelar pago',
        'success-title': 'Pago procesado',
        'success-message': 'Tu pago ha sido recibido correctamente',
        'cancel-confirm': '¿Estás seguro que querés cancelar la solicitud?',
        'cancel-warning': 'Cancelaciones frecuentes pueden limitar tu capacidad para realizar nuevos pagos.',
        'cancel-reason-label': 'Por favor, indicá el motivo para cancelar la solicitud',
        'cancel-reason-1': 'Quiero recargar un monto diferente',
        'cancel-reason-2': 'No se puede transferir',
        'cancel-reason-3': 'Cambié de opinión sobre el pago',
        'cancel-reason-4': 'Otro',
        'cancel-back': 'Volver atrás',
        'cancel-submit': 'Cancelar solicitud',
        'cancel-specify-reason': 'Indicá tu motivo',
        'appeal-title': 'Apelación de Pago',
        'appeal-subtitle': 'Completa el formulario de apelación',
        'appeal-id-label': 'ID de Pago',
        'appeal-amount-label': 'Monto total transferido',
        'appeal-receipt-label': 'Comprobantes de pago',
        'appeal-upload-text': 'Subir archivo(s) de comprobante',
        'appeal-pdf-hint': 'Adjuntar solo archivos PDF',
        'appeal-submit': 'Hacer Apelación',
    },
    en: {
        'processing-title': 'Payment with Mercado Pago',
        'processing-subtitle': 'Transfer',
        'recipient': 'Recipient',
        'account': 'CVU/CBU',
        'amount': 'Exact amount',
        'after-payment': 'After payment, tap',
        'after-payment-pagado': '«I Paid»:',
        'paid-button': 'I Paid',
        'time-to-pay': 'Time to pay',
        'confirm-title': 'Attention',
        'confirm-message': 'Verify the information and confirm if you paid correctly',
        'amount-label': 'Payment amount',
        'phone-label': 'Recipient phone',
        'bank-label': 'Bank',
        'recipient-full': 'Recipient full name',
        'back-button': 'Back to payment',
        'confirm-button': 'Confirm',
        'searching-title': 'We are looking for your transfer',
        'searching-message': 'The process can take up to 20 minutes.',
        'back-to-store': 'Back to store',
        'back-to-payment': 'Back to payment',
        'payment-details': 'Payment details',
        'detail-id': 'ID',
        'detail-bank': 'Bank details',
        'detail-method': 'Payment method',
        'detail-amount': 'Amount',
        'detail-recipient': 'Recipient',
        'detail-bank-name': 'Recipient bank',
        'detail-date': 'Created date',
        'not-credited': 'Not credited',
        'expired-message': 'You cancelled the payment, or payment time has expired',
        'appeal-button': 'Claim Payment',
        'appeal-text': 'Claim Payment',
        'success-title': 'Payment processed',
        'success-message': 'Your payment has been received successfully',
        'cancel-confirm': 'Are you sure you want to cancel the request?',
        'cancel-warning': 'Frequent cancellations may limit your ability to make new payments.',
        'cancel-reason-label': 'Please indicate the reason for canceling the request',
        'cancel-reason-1': 'I want to reload a different amount',
        'cancel-reason-2': 'Cannot transfer',
        'cancel-reason-3': 'Changed my mind about the payment',
        'cancel-reason-4': 'Other',
        'cancel-back': 'Go back',
        'cancel-submit': 'Cancel request',
        'cancel-specify-reason': 'Specify your reason',
        'appeal-title': 'Payment Appeal',
        'appeal-subtitle': 'Fill out the appeal form',
        'appeal-id-label': 'Payment ID',
        'appeal-amount-label': 'Total amount of funds transferred',
        'appeal-receipt-label': 'Payment receipts',
        'appeal-upload-text': 'Upload receipt file(s)',
        'appeal-pdf-hint': 'Attach only pdf files',
        'appeal-submit': 'Make Appeal',
    }
};

const FLAG_SRC = {
    es: "data:image/svg+xml,%3csvg%20width='20'%20height='15'%20viewBox='0%200%2020%2015'%20fill='none'%20xmlns='http://www.w3.org/2000/svg'%3e%3cg%20clip-path='url(%23clip0_11349_37254)'%3e%3cg%20clip-path='url(%23clip1_11349_37254)'%3e%3crect%20width='20'%20height='15'%20fill='white'/%3e%3cpath%20fill-rule='evenodd'%20clip-rule='evenodd'%20d='M0%200V15H20V0H0Z'%20fill='%23F7FCFF'/%3e%3cmask%20id='mask0_11349_37254'%20style='mask-type:luminance'%20maskUnits='userSpaceOnUse'%20x='0'%20y='0'%20width='20'%20height='15'%3e%3cpath%20fill-rule='evenodd'%20clip-rule='evenodd'%20d='M0%200V15H20V0H0Z'%20fill='white'/%3e%3c/mask%3e%3cg%20mask='url(%23mask0_11349_37254)'%3e%3cpath%20fill-rule='evenodd'%20clip-rule='evenodd'%20d='M0%200V5H20V0H0Z'%20fill='%2358A5FF'/%3e%3cpath%20fill-rule='evenodd'%20clip-rule='evenodd'%20d='M0%2010V15H20V10H0Z'%20fill='%2358A5FF'/%3e%3cpath%20fill-rule='evenodd'%20clip-rule='evenodd'%20d='M10.3456%209.05472C10.3456%209.05472%209.7069%2010.3789%209.14556%2010.6504C9.38786%2010.0431%209.47704%208.80384%209.47704%208.80384C9.47704%208.80384%208.01959%209.38168%207.50805%209.27637C8.12316%208.84653%208.97696%208.05039%208.97696%208.05039C8.97696%208.05039%207.09057%207.43368%207.13685%207.17122C7.9884%207.32397%209.15947%207.15745%209.15947%207.15745C9.15947%207.15745%207.8361%205.57043%207.97031%205.45241C8.17202%205.64848%209.758%206.47693%209.758%206.47693C9.758%206.47693%209.87306%205.06493%2010.2144%204.58887C10.2553%204.92491%2010.7344%206.44639%2010.7344%206.44639C10.7344%206.44639%2011.6974%205.48279%2012.232%205.48279C11.9972%205.77389%2011.4412%207.0526%2011.4412%207.0526C11.4412%207.0526%2012.8267%207.0308%2013.3534%207.29011C12.7154%207.38087%2011.5975%207.94206%2011.5975%207.94206C11.5975%207.94206%2013.0534%209.04646%2012.9207%209.27637C12.139%208.89398%2011.2353%208.76592%2011.2353%208.76592C11.2353%208.76592%2011.4851%2010.3028%2011.2832%2010.6504C11.0863%2010.1388%2010.3456%209.05472%2010.3456%209.05472Z'%20fill='%23FFD018'%20stroke='%23F19900'%20stroke-opacity='0.98'%20stroke-width='0.5'/%3e%3cpath%20fill-rule='evenodd'%20clip-rule='evenodd'%20d='M10.2617%208.33887C10.6069%208.33887%2010.8867%208.05905%2010.8867%207.71387C10.8867%207.36869%2010.6069%207.08887%2010.2617%207.08887C9.91654%207.08887%209.63672%207.36869%209.63672%207.71387C9.63672%208.05905%209.91654%208.33887%2010.2617%208.33887Z'%20fill='%23FFD018'%20stroke='%23F19900'%20stroke-opacity='0.98'%20stroke-width='0.5'/%3e%3c/g%3e%3c/g%3e%3c/g%3e%3crect%20x='0.5'%20y='0.5'%20width='19'%20height='14'%20rx='1'%20stroke='black'%20stroke-opacity='0.1'%20style='mix-blend-mode:multiply'/%3e%3cdefs%3e%3cclipPath%20id='clip0_11349_37254'%3e%3crect%20width='20'%20height='15'%20rx='1.5'%20fill='white'/%3e%3c/clipPath%3e%3cclipPath%20id='clip1_11349_37254'%3e%3crect%20width='20'%20height='15'%20fill='white'/%3e%3c/clipPath%3e%3c/defs%3e%3c/svg%3e",
    en: "data:image/svg+xml,%3csvg%20width='20'%20height='15'%20viewBox='0%200%2020%2015'%20fill='none'%20xmlns='http://www.w3.org/2000/svg'%3e%3cg%20clip-path='url(%23clip0_11349_37244)'%3e%3cg%20clip-path='url(%23clip1_11349_37244)'%3e%3crect%20width='20'%20height='15'%20fill='white'/%3e%3cpath%20fill-rule='evenodd'%20clip-rule='evenodd'%20d='M0%200V15H20V0H0Z'%20fill='%232E42A5'/%3e%3cmask%20id='mask0_11349_37244'%20style='mask-type:luminance'%20maskUnits='userSpaceOnUse'%20x='0'%20y='0'%20width='20'%20height='15'%3e%3cpath%20fill-rule='evenodd'%20clip-rule='evenodd'%20d='M0%200V15H20V0H0Z'%20fill='white'/%3e%3c/mask%3e%3cg%20mask='url(%23mask0_11349_37244)'%3e%3cpath%20d='M-2.22754%2013.9283L2.17352%2015.7896L20.0993%202.02354L22.4207%20-0.742388L17.7145%20-1.36448L10.403%204.56768L4.51795%208.56458L-2.22754%2013.9283Z'%20fill='white'/%3e%3cpath%20d='M-1.625%2015.2324L0.617135%2016.3125L21.5872%20-0.999338H18.4389L-1.625%2015.2324Z'%20fill='%23F50100'/%3e%3cpath%20d='M22.2266%2013.9283L17.8255%2015.7896L-0.100267%202.02354L-2.42172%20-0.742388L2.28457%20-1.36448L9.59607%204.56768L15.4811%208.56458L22.2266%2013.9283Z'%20fill='white'/%3e%3cpath%20d='M22.0762%2014.8642L19.834%2015.9443L10.9048%208.53224L8.25745%207.7041L-2.64532%20-0.732895H0.502899L11.3995%207.50384L14.2939%208.49687L22.0762%2014.8642Z'%20fill='%23F50100'/%3e%3cmask%20id='path-9-inside-1_11349_37244'%20fill='white'%3e%3cpath%20d='M12.3604%205H21.2666V10H12.3604V16.25H7.63867V10H-1.2334V5H7.63867V-1.25H12.3604V5Z'/%3e%3c/mask%3e%3cpath%20d='M12.3604%205H21.2666V10H12.3604V16.25H7.63867V10H-1.2334V5H7.63867V-1.25H12.3604V5Z'%20fill='%23F50100'/%3e%3cpath%20d='M12.3604%205H11.1104V6.25H12.3604V5ZM21.2666%205H22.5166V3.75H21.2666V5ZM21.2666%2010V11.25H22.5166V10H21.2666ZM12.3604%2010V8.75H11.1104V10H12.3604ZM12.3604%2016.25V17.5H13.6104V16.25H12.3604ZM7.63867%2016.25H6.38867V17.5H7.63867V16.25ZM7.63867%2010H8.88867V8.75H7.63867V10ZM-1.2334%2010H-2.4834V11.25H-1.2334V10ZM-1.2334%205V3.75H-2.4834V5H-1.2334ZM7.63867%205V6.25H8.88867V5H7.63867ZM7.63867%20-1.25V-2.5H6.38867V-1.25H7.63867ZM12.3604%20-1.25H13.6104V-2.5H12.3604V-1.25ZM12.3604%205V6.25H21.2666V5V3.75H12.3604V5ZM21.2666%205H20.0166V10H21.2666H22.5166V5H21.2666ZM21.2666%2010V8.75H12.3604V10V11.25H21.2666V10ZM12.3604%2010H11.1104V16.25H12.3604H13.6104V10H12.3604ZM12.3604%2016.25V15H7.63867V16.25V17.5H12.3604V16.25ZM7.63867%2016.25H8.88867V10H7.63867H6.38867V16.25H7.63867ZM7.63867%2010V8.75H-1.2334V10V11.25H7.63867V10ZM-1.2334%2010H0.0166016V5H-1.2334H-2.4834V10H-1.2334ZM-1.2334%205V6.25H7.63867V5V3.75H-1.2334V5ZM7.63867%205H8.88867V-1.25H7.63867H6.38867V5H7.63867ZM7.63867%20-1.25V0H12.3604V-1.25V-2.5H7.63867V-1.25ZM12.3604%20-1.25H11.1104V5H12.3604H13.6104V-1.25H12.3604Z'%20fill='white'%20mask='url(%23path-9-inside-1_11349_37244)'/%3e%3c/g%3e%3c/g%3e%3c/g%3e%3crect%20x='0.5'%20y='0.5'%20width='19'%20height='14'%20rx='1'%20stroke='black'%20stroke-opacity='0.1'%20style='mix-blend-mode:multiply'/%3e%3cdefs%3e%3cclipPath%20id='clip0_11349_37244'%3e%3crect%20width='20'%20height='15'%20rx='1.5'%20fill='white'/%3e%3c/clipPath%3e%3cclipPath%20id='clip1_11349_37244'%3e%3crect%20width='20'%20height='15'%20fill='white'/%3e%3c/clipPath%3e%3c/defs%3e%3c/svg%3e"
};

const CHECK_SVG = `<svg xmlns="http://www.w3.org/2000/svg" width="1em" height="1em"
  viewBox="0 0 33 32" fill="none"><path d="M27.1673 8L12.5007 22.6667L5.83398 16"
  stroke="currentColor" stroke-width="3" stroke-linecap="round"
  stroke-linejoin="round"></path></svg>`;

function svgFromString(svgStr) {
    const tpl = document.createElement('template');
    tpl.innerHTML = svgStr.trim();
    return tpl.content.firstChild;
}

class PaymentApp {
    constructor() {
        this.currentLanguage = 'es';
        this.currentState = 'processing';
        this.transactionData = null;
        this.timerInterval = null;
        this.statusCheckInterval = null;
        this.timeRemaining = 500;
        this.timeTotal = this.timeRemaining;
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.loadTransactionData();
        this.updateLanguage('es');
        this.startTimer();
        this.initLanguageDropdown();
        this.initOrUpdateCancelListeners();
        this.initPaymentAccordions();
    }

    initLanguageDropdown() {
        this.langDropdown = document.querySelector('.lang-dropdown');
        this.langButton = document.querySelector('.lang-button');
        this.langMenu = document.querySelector('.lang-menu');

        if (!this.langDropdown || !this.langButton || !this.langMenu) return;

        this.langButton.addEventListener('click', (e) => {
            e.stopPropagation();
            this.langMenu.classList.toggle('open');
            this.langButton.classList.toggle('open'); // for arrow rotation
        });

        this.langMenu.querySelectorAll('[data-lang]').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.stopPropagation();
                const lang = e.currentTarget.dataset.lang;
                this.updateLanguage(lang); // this will also close the menu
            });
        });

        document.addEventListener('click', () => {
            this.langMenu.classList.remove('open');
            this.langButton.classList.remove('open');
        });
    }

    initTimerIcon() {
        const svg = document.querySelector('.timer-icon svg');
        if (!svg) return;

        this.progressCircle = svg.querySelector('circle:nth-of-type(2)');
        if (!this.progressCircle) return;

        const r = parseFloat(this.progressCircle.getAttribute('r') || '13');
        this.circumference = 2 * Math.PI * r;

        this.progressCircle.style.strokeDasharray = `${this.circumference}px`;
        this.progressCircle.style.strokeDashoffset = '0px';
    }

    initPaymentAccordions() {
        const isMobile = () => window.matchMedia('(max-width: 768px)').matches;

        document.querySelectorAll('.payment-details-accordion').forEach((wrap) => {
            const header = wrap.querySelector('.accordion-header');
            const content = wrap.querySelector('.accordion-content');
            const icon   = wrap.querySelector('.accordion-icon');

            if (!header || !content) return;

            header.setAttribute('aria-expanded', 'false');
            content.setAttribute('aria-hidden', 'true');

            header.addEventListener('click', (e) => {
                e.preventDefault();

                const expanded = header.getAttribute('aria-expanded') === 'true';
                const next = !expanded;

                header.setAttribute('aria-expanded', String(next));
                content.setAttribute('aria-hidden', String(!next));
                content.classList.toggle('hidden', !next);

                if (icon) icon.classList.toggle('open', next);

                if (next && isMobile()) {
                    requestAnimationFrame(() => {
                        wrap.scrollIntoView({ behavior: 'smooth', block: 'start' });
                    });
                }
            });
        });
    }

    updateTimerVisual() {
        if (!this.progressCircle || !this.circumference || !this.timeTotal) return;

        const fractionLeft = Math.max(0, Math.min(1, this.timeRemaining / this.timeTotal));
        const dashoffset = this.circumference * (1 - fractionLeft);
        this.progressCircle.style.strokeDashoffset = `${dashoffset}px`;
    }

    setupEventListeners() {
        document.getElementById('confirm-payment').addEventListener('click', () => this.showState('confirmation'));
        document.getElementById('close-modal').addEventListener('click', () => this.showState('processing'));
        document.getElementById('cancel-payment').addEventListener('click', () => this.showState('processing'));
        document.getElementById('submit-payment').addEventListener('click', () => this.submitPayment());

        document.getElementById('return-to-store').addEventListener('click', () => this.redirect());
        document.getElementById('back-to-payment').addEventListener('click', () => this.showState('processing'));

        document.getElementById('return-to-store-expired').addEventListener('click', () => this.redirect());
        document.getElementById('appeal-payment').addEventListener('click', () => this.showState('appeal'));

        document.getElementById('return-to-store-success').addEventListener('click', () => this.redirect());

        document.getElementById('open-cancel-modal').addEventListener('click', () => this.openCancelModal());
        document.getElementById('close-cancel-modal').addEventListener('click', () => this.closeCancelModal());
        document.getElementById('cancel-back-btn').addEventListener('click', () => this.closeCancelModal());
        document.getElementById('cancel-back-btn-2').addEventListener('click', () => this.showCancelSlide(1));
        document.getElementById('cancel-confirm-btn').addEventListener('click', () => this.handleCancelReason());
        document.getElementById('cancel-submit-other').addEventListener('click', () => this.submitCancellation());

        document.querySelectorAll('input[name="cancelReason"]').forEach(radio => {
            radio.addEventListener('change', (e) => {
                if (e.target.value === 'other') {
                    this.showCancelSlide(2);
                }
            });
        });

        document.getElementById('cancel-comment').addEventListener('input', (e) => {
            const submitBtn = document.getElementById('cancel-submit-other');
            submitBtn.disabled = e.target.value.trim() === '';
        });

        // Appeal form
        document.getElementById('back-from-appeal').addEventListener('click', () => this.showState('expired'));
        document.getElementById('file-upload-area').addEventListener('click', () => {
            document.getElementById('appeal-file').click();
        });
        document.getElementById('file-upload-area').addEventListener('dragover', (e) => {
            e.preventDefault();
            e.currentTarget.classList.add('dragover');
        });
        document.getElementById('file-upload-area').addEventListener('dragleave', (e) => {
            e.currentTarget.classList.remove('dragover');
        });
        document.getElementById('file-upload-area').addEventListener('drop', (e) => {
            e.preventDefault();
            e.currentTarget.classList.remove('dragover');
            this.handleFileUpload(e.dataTransfer.files);
        });
        document.getElementById('appeal-file').addEventListener('change', (e) => {
            this.handleFileUpload(e.target.files);
        });
        document.getElementById('appeal-form').addEventListener('submit', (e) => {
            e.preventDefault();
            this.submitAppeal();
        });

        // Accordion toggles
        document.getElementById('toggle-details')?.addEventListener('click', () => this.toggleAccordion('details-content'));
        document.getElementById('toggle-details-expired')?.addEventListener('click', () => this.toggleAccordion('details-content-expired'));

        // Copy buttons
        document.querySelectorAll('.copy-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const valueElement = e.target.closest('.detail-row')?.querySelector('.detail-val') ||
                    e.target.closest('.info-row')?.querySelector('.info-value') ||
                    e.target.closest('.confirmation-details-container')?.querySelector('.detail-value') ||
                e.target.parentElement;
                const textToCopy = valueElement?.textContent?.trim() || '';
                this.copyToClipboard(textToCopy);
            });
        });

        document.addEventListener('click', async (e) => {
            const btn = e.target.closest('.copy-btn');
            if (!btn) return;

            // Find value in the closest known container
            const container = btn.closest('.detail-row, .info-row, .confirmation-details-container');
            const valueEl =
                container?.querySelector('.detail-val, .info-value, .detail-value') ||
                document.getElementById('confirm-amount') ||
                document.getElementById('confirm-phone') ||
                document.getElementById('confirm-bank') ||
                document.getElementById('confirm-recipient');

            const textToCopy = (valueEl?.textContent || '').trim();
            if (!textToCopy) return;

            // Copy
            try {
                if (navigator.clipboard?.writeText) {
                    await navigator.clipboard.writeText(textToCopy);
                } else {
                    const ta = document.createElement('textarea');
                    ta.value = textToCopy;
                    ta.style.position = 'fixed';
                    ta.style.opacity = '0';
                    document.body.appendChild(ta);
                    ta.select();
                    document.execCommand('copy');
                    document.body.removeChild(ta);
                }
            } catch {
            }

            const existingSvg = btn.querySelector('svg');
            if (existingSvg) {
                // Cache original once
                if (!btn.dataset.iconOriginal) {
                    btn.dataset.iconOriginal = existingSvg.outerHTML;
                }
                existingSvg.replaceWith(svgFromString(CHECK_SVG));
                btn.classList.add('copied');

                setTimeout(() => {
                    const currentSvg = btn.querySelector('svg');
                    if (btn.dataset.iconOriginal && currentSvg) {
                        currentSvg.replaceWith(svgFromString(btn.dataset.iconOriginal));
                    }
                    btn.classList.remove('copied');
                }, 1000);
            }
        });
    }

    loadTransactionData() {
        // Simulated transaction data from widget-details.php
        this.transactionData = {
            order_id: "3XU91ACAR1UUWGA0",
            currency: "ARS",
            iban: "BSB",
            status: "PROCESSING", // Can be PROCESSING, PENDING, PAID, EXPIRED, CANCELLED
            transactionId: "3XU91ACAR1UUWGA0",
            transaction_id: "U2FsdGVkX1+qJB3bSEFrRgmOE83y0HrwJJfQT3QAvv0=",
            redirect_url: "https://example.com",
            amount: "10.0000000000",
            pocket_address: "4012312344314854",
            card: "4012312344314854",
            operator_bank: 203,
            operator_bank_title: "Westpac .",
            customer_id: "1AY008V66WAA94KC",
            user_id: 6,
            alias: "BSB ",
            send_id: "UB Company Ltd.",
            expires_at: "2025-09-05T08:59:26.201Z",
            first_name: "Nam",
            last_name: "Lastov",
            service: 0,
            merchantBank: {
                bank: {
                    id: 19,
                    title: "Other Bank"
                }
            },
            priority_bank: null,
            tax_id: "123123132"
        };

        this.populateTransactionDetails();
    }

    populateTransactionDetails() {
        const data = this.transactionData;
        const formattedAmount = this.formatAmount(data.amount, data.currency);

        // Processing state
        document.getElementById('recipient-name').textContent = `${data.first_name} ${data.last_name}`;
        document.getElementById('account-number').textContent = data.card;
        document.getElementById('amount-display').textContent = formattedAmount;

        // Confirmation state
        document.getElementById('confirm-amount').textContent = formattedAmount;
        document.getElementById('confirm-phone').textContent = data.iban || '***';
        document.getElementById('confirm-bank').textContent = data.operator_bank_title;
        document.getElementById('confirm-recipient').textContent = `${data.first_name} ${data.last_name}`;

        // Loading state details
        document.getElementById('detail-id').textContent = data.transaction_id.substring(0, 20) + '...';
        document.getElementById('detail-bank').textContent = data.card;
        document.getElementById('detail-method').textContent = 'Pago con Mercado Pago';
        document.getElementById('detail-amount').textContent = formattedAmount;
        document.getElementById('detail-recipient').textContent = `${data.first_name} ${data.last_name}`;
        document.getElementById('detail-bank-name').textContent = data.operator_bank_title;
        document.getElementById('detail-date').textContent = new Date(data.expires_at).toLocaleString();

        // Expired state details (same as loading)
        document.getElementById('detail-id-exp').textContent = data.transaction_id.substring(0, 20) + '...';
        document.getElementById('detail-bank-exp').textContent = data.card;
        document.getElementById('detail-method-exp').textContent = 'Pago con Mercado Pago';
        document.getElementById('detail-amount-exp').textContent = formattedAmount;
        document.getElementById('detail-recipient-exp').textContent = `${data.first_name} ${data.last_name}`;
        document.getElementById('detail-bank-name-exp').textContent = data.operator_bank_title;
        document.getElementById('detail-date-exp').textContent = new Date(data.expires_at).toLocaleString();

        // Expired amount display
        document.getElementById('expired-amount').textContent = formattedAmount;
        document.getElementById('success-amount').textContent = formattedAmount;

        // Store amount in account copy
        document.getElementById('copy-account').addEventListener('click', () => {
            this.copyToClipboard(data.card);
        });
    }

    formatAmount(amount, currency) {
        const num = parseFloat(amount);
        return `${num.toLocaleString('es-AR', { minimumFractionDigits: 2, maximumFractionDigits: 2 })} ${currency}`;
    }

    updateLanguage(lang) {
        this.currentLanguage = lang;

        // Update language button
        document.querySelectorAll('[data-lang]').forEach(btn => {
            btn.classList.toggle('active', btn.dataset.lang === lang);
        });

        // Update all translatable elements
        document.querySelectorAll('[data-i18n]').forEach(el => {
            const key = el.dataset.i18n;
            const text = translations[lang][key] || translations.es[key];
            el.textContent = text;
        });

        // Update lang dropdown button
        const activeBtn = document.querySelector(`[data-lang="${lang}"]`);
        if (activeBtn) {
            const dropdownBtn = document.querySelector('.lang-button');
            const flag = activeBtn.querySelector('.flag')?.textContent || '';
            const langName = lang === 'es' ? 'Español' : 'English';
            dropdownBtn.innerHTML = `<span class="flag">${flag}</span><span>${langName}</span>`;
        }

        const newLabel = lang === 'es' ? 'Español' : 'English';

        const flagImg = document.querySelector('.lang-button .flag img');
        if (flagImg) {
            flagImg.src = FLAG_SRC[lang];
            flagImg.alt = newLabel;
        } else {
            // If img is missing, create it
            const flagWrap = document.querySelector('.lang-button .flag');
            if (flagWrap) {
                const img = document.createElement('img');
                img.src = FLAG_SRC[lang];
                img.alt = newLabel;
                flagWrap.appendChild(img);
            }
        }

        // Optional: visually mark the selected option in the dropdown
        document.querySelectorAll('.lang-menu .lang-option').forEach(btn => {
            btn.classList.toggle('active', btn.dataset.lang === lang);
        });

        // Close menu as before
        const menu = document.querySelector('.lang-menu');
        const btn = document.querySelector('.lang-button');
        menu?.classList.remove('open');
        btn?.classList.remove('open');
    }

    showState(state) {
        // Hide all states
        document.querySelectorAll('.state-card').forEach(card => {
            card.classList.add('hidden');
        });

        // Show selected state
        const stateElement = document.getElementById(`state-${state}`);
        if (stateElement) {
            stateElement.classList.remove('hidden');
            this.currentState = state;

            // Populate appeal form if showing appeal state
            if (state === 'appeal') {
                this.populateAppealForm();
            }
        }
    }

    submitPayment() {
        // Call start-transaction API
        const formData = {
            transaction_id: this.transactionData.transaction_id,
            utr: this.transactionData.first_name,
            tax_id: this.transactionData.tax_id
        };

        fetch('./start-transaction.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(formData)
        })
            .then(response => response.json())
            .then(data => {
                this.showState('loading');
                this.startStatusChecking();
            })
            .catch(error => {
                console.error('Payment submission error:', error);
                this.showState('loading');
            });
    }

    startStatusChecking() {
        // Check transaction status every 5 seconds
        this.statusCheckInterval = setInterval(() => {
            this.checkTransactionStatus();
        }, 5000);
    }

    checkTransactionStatus() {
        const formData = {
            transaction_id: this.transactionData.transaction_id,
            user_id: this.transactionData.user_id
        };

        fetch('./check-transaction-status.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(formData)
        })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'PAID') {
                    clearInterval(this.statusCheckInterval);
                    this.showState('success');
                    setTimeout(() => this.redirect(), 3000);
                } else if (data.status === 'EXPIRED' || data.status === 'CANCELLED') {
                    clearInterval(this.statusCheckInterval);
                    this.showState('expired');
                }
            })
            .catch(error => console.error('Status check error:', error));
    }

    startTimer() {
        const timerElement = document.getElementById('timer');

        this.initTimerIcon();
        this.updateTimerVisual();

        this.timerInterval = setInterval(() => {
            this.timeRemaining--;

            if (this.timeRemaining <= 0) {
                clearInterval(this.timerInterval);
                this.timeRemaining = 0;
                this.updateTimerVisual(); // ensure ring fully empty
                this.transactionData.status = 'EXPIRED';
                this.showState('expired');
                return;
            }

            const minutes = Math.floor(this.timeRemaining / 60);
            const seconds = this.timeRemaining % 60;
            timerElement.textContent = `${minutes} min ${seconds} seg`;

            this.updateTimerVisual();
        }, 1000);
    }

    toggleAccordion(contentId) {
        const content = document.getElementById(contentId);
        content.classList.toggle('hidden');
    }

    copyToClipboard(text) {
        if (!text) return;

        navigator.clipboard.writeText(text).then(() => {
            // Show feedback
            const feedback = document.createElement('div');
            feedback.className = 'copy-feedback';
            feedback.textContent = this.currentLanguage === 'es' ? 'Copiado!' :
                this.currentLanguage === 'en' ? 'Copied!' : 'Скопировано!';
            document.body.appendChild(feedback);

            setTimeout(() => feedback.remove(), 2000);
        }).catch(err => {
            console.error('Copy failed:', err);
        });
    }

    appealPayment() {
        alert('Payment appeal initiated. Check your email for next steps.');
    }

    redirect() {
        const redirectUrl = localStorage.getItem('redirectUrl') || this.transactionData.redirect_url;
        if (redirectUrl) {
            window.location.href = redirectUrl;
        }
    }

    openCancelModal() {
        document.getElementById('cancel-modal').classList.remove('hidden');
        this.showCancelSlide(1);
    }

    showCancelSlide(slideNumber) {
        const slides = Array.from(document.querySelectorAll('.cancel-slide'));
        const next = document.getElementById(`cancel-slide-${slideNumber}`);
        if (!next) return;

        const current = slides.find(s => !s.classList.contains('hidden'));

        if (current === next) return;

        next.classList.remove('hidden');
        next.classList.add('slide-in-right');

        if (current) {
            current.classList.add('slide-out-left');
            // after the animation, fully hide current
            setTimeout(() => {
                current.classList.add('hidden');
                current.classList.remove('slide-out-left');
            }, 250);
        }

        // animate in the next slide
        // (allowing a tick so the browser applies the starting transform)
        requestAnimationFrame(() => {
            next.classList.add('show');
            setTimeout(() => {
                next.classList.remove('slide-in-right', 'show');
            }, 250);
        });
    }

    closeCancelModal() {
        document.getElementById('cancel-modal').classList.add('hidden');
    }

    handleCancelReason() {
        const selected = document.querySelector('input[name="cancelReason"]:checked')?.value;
        if (selected === 'other') this.showCancelSlide(2);
        else this.submitCancellation();
    }

    initOrUpdateCancelListeners() {
        document.querySelectorAll('input[name="cancelReason"]').forEach(radio => {
            radio.removeEventListener('_cancelOther', radio._cancelOtherHandler || (()=>{}));
            radio._cancelOtherHandler = (e) => {
                if (e.target.value === 'other') this.showCancelSlide(2);
            };
            radio.addEventListener('change', radio._cancelOtherHandler);
        });

        const comment = document.getElementById('cancel-comment');
        if (comment) {
            comment.addEventListener('input', (e) => {
                const submitBtn = document.getElementById('cancel-submit-other');
                if (submitBtn) submitBtn.disabled = e.target.value.trim() === '';
            });
        }
    }

    submitCancellation() {
        const selectedReason = document.querySelector('input[name="cancelReason"]:checked')?.value;
        const comment = document.getElementById('cancel-comment')?.value;

        const cancelData = {
            transaction_id: this.transactionData.transaction_id,
            reason: selectedReason,
            comment: comment || '',
            timestamp: new Date().toISOString()
        };

        console.log('Payment cancelled:', cancelData);

        this.closeCancelModal();
        this.redirect();
    }

    populateAppealForm() {
        const data = this.transactionData;
        document.getElementById('appeal-id').value = data.transaction_id.substring(0, 40) + '...';
        document.getElementById('appeal-amount').value = this.formatAmount(data.amount, '');
    }

    handleFileUpload(files) {
        const fileList = document.getElementById('file-list');
        fileList.innerHTML = '';

        const validFiles = [];
        for (let file of files) {
            if (file.type === 'application/pdf') {
                validFiles.push(file);
            }
        }

        if (validFiles.length === 0) {
            const error = document.createElement('div');
            error.className = 'file-error';
            error.textContent = this.currentLanguage === 'es' ?
                'Por favor, selecciona solo archivos PDF' :
                this.currentLanguage === 'ru' ?
                    'Пожалуйста, выберите только файлы PDF' :
                    'Please select only PDF files';
            fileList.appendChild(error);
            return;
        }

        validFiles.forEach((file, index) => {
            const fileItem = document.createElement('div');
            fileItem.className = 'file-item';
            fileItem.innerHTML = `
                <span class="file-name"><svg viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
  <path d="M10.5 1.70215V4.80005C10.5 5.22009 10.5 5.43011 10.5817 5.59055C10.6537 5.73167 10.7684 5.8464 10.9095 5.91831C11.0699 6.00005 11.28 6.00005 11.7 6.00005H14.7979M10.5 12.75H6M12 9.75H6M15 7.49117V12.9C15 14.1601 15 14.7902 14.7548 15.2715C14.539 15.6948 14.1948 16.039 13.7715 16.2548C13.2902 16.5 12.6601 16.5 11.4 16.5H6.6C5.33988 16.5 4.70982 16.5 4.22852 16.2548C3.80516 16.039 3.46095 15.6948 3.24524 15.2715C3 14.7902 3 14.1601 3 12.9V5.1C3 3.83988 3 3.20982 3.24524 2.72852C3.46095 2.30516 3.80516 1.96095 4.22852 1.74524C4.70982 1.5 5.33988 1.5 6.6 1.5H9.00883C9.55916 1.5 9.83432 1.5 10.0933 1.56217C10.3229 1.61729 10.5423 1.7082 10.7436 1.83156C10.9707 1.9707 11.1653 2.16527 11.5544 2.55442L13.9456 4.94558C14.3347 5.33473 14.5293 5.5293 14.6684 5.75636C14.7918 5.95767 14.8827 6.17715 14.9378 6.40673C15 6.66568 15 6.94084 15 7.49117Z" stroke="#241f4d" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>
 ${file.name}</span>
                <button type="button" class="file-remove" data-index="${index}">✕</button>
            `;
            fileList.appendChild(fileItem);

            fileItem.querySelector('.file-remove').addEventListener('click', (e) => {
                e.preventDefault();
                fileItem.remove();
            });
        });

        // Store files for submission
        this.uploadedFiles = validFiles;
    }

    submitAppeal() {
        if (!this.uploadedFiles || this.uploadedFiles.length === 0) {
            alert(this.currentLanguage === 'es' ?
                'Por favor, adjunta al menos un archivo PDF' :
                this.currentLanguage === 'ru' ?
                    'Пожалуйста, прикрепите хотя бы один файл PDF' :
                    'Please attach at least one PDF file');
            return;
        }

        const appealData = {
            transaction_id: this.transactionData.transaction_id,
            user_id: this.transactionData.user_id,
            amount: this.transactionData.amount,
            files_count: this.uploadedFiles.length,
            timestamp: new Date().toISOString()
        };

        console.log('Appeal submitted:', appealData);

        // In production, upload files and submit form to your API
        // For now, show success message
        alert(this.currentLanguage === 'es' ?
            'Apelación enviada correctamente. Te contactaremos pronto.' :
            this.currentLanguage === 'ru' ?
                'Апелляция успешно подана. Мы скоро свяжемся с вами.' :
                'Appeal submitted successfully. We will contact you soon.');

        this.redirect();
    }
}

// Initialize app when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    new PaymentApp();
});
